<?php declare(strict_types=1);

namespace DHLParcel\Shipping;

use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Plugin;
use Shopware\Core\System\CustomField\CustomFieldTypes;
use Shopware\Core\Framework\Plugin\Context\InstallContext;
use Shopware\Core\Framework\Plugin\Context\UpdateContext;
use Shopware\Core\Framework\Plugin\Context\UninstallContext;
use DHLParcel\Shipping\Service\ServicePointService;

class DHLParcelShipping extends Plugin
{
    public function install(InstallContext $installContext): void
    {
        parent::install($installContext);

        // Install Servicepoint Custom Field
        $this->createServicePointFields($installContext);
    }

    public function update(UpdateContext $updateContext): void
    {
        parent::update($updateContext);

        // Install Servicepoint Custom Field, introduced in 1.2.0
        if (version_compare($updateContext->getCurrentPluginVersion(), '1.2.0', '<')) {
            $this->createServicePointFields($updateContext);
        }
    }

    public function uninstall(UninstallContext $uninstallContext): void
    {
        parent::uninstall($uninstallContext);

        $this->removeServicePointFields($uninstallContext);
    }

    /**
     * @param InstallContext $context
     */
    protected function createServicePointFields(InstallContext $context): void
    {
        /** @var EntityRepositoryInterface $customFieldSetRepository */
        $customFieldSetRepository = $this->container->get('custom_field_set.repository');

        $customFieldSetRepository->create([[
            'name' => ServicePointService::CUSTOM_FIELD_SERVICEPOINT_ID,
            'config' => [
                'label' => [
                    'en-GB' => 'DHLParcel ServicePoint ID',
                    'de-DE' => 'DHLParcel ServicePoint ID',
                    'nl-NL' => 'DHLParcel ServicePoint ID',
                ],
                'translated' => true,
            ],
            'customFields' => [
                [
                    'name'      => ServicePointService::CUSTOM_FIELD_SERVICEPOINT_ID,
                    'type'      => CustomFieldTypes::TEXT
                ],
            ],
            'relations' => [
                [
                    'entityName' => 'order',
                ],
            ],
        ]], $context->getContext());
    }

    protected function removeServicePointFields(UninstallContext $context): void
    {
        /** @var EntityRepositoryInterface $customFieldSetRepository */
        $customFieldSetRepository = $this->container->get('custom_field_set.repository');

        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('name', ServicePointService::CUSTOM_FIELD_SERVICEPOINT_ID));
        $ids = $customFieldSetRepository->searchIds($criteria, $context->getContext())->getIds();

        if ($ids) {
            $customFieldSetRepository->delete([['id' => $ids[0]]], $context->getContext());
        }
    }
}
