<?php declare(strict_types = 1);

namespace DHLParcel\Shipping\Service;

use Psr\Log\LoggerInterface;
use Shopware\Core\Checkout\Order\OrderEntity;
use GuzzleHttp\Client;

class WebhookService
{
    /**
     * @var Client
     */
    protected $httpClient;

    /**
     * @var ConfigService
     */
    protected $configService;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * WebhookService constructor.
     * @param ConfigService $configService
     */
    public function __construct(
        ConfigService $configService,
        LoggerInterface $logger
    ) {
        $this->configService = $configService;
        $this->httpClient = new Client();
        $this->logger = $logger;
    }

    /**
     * @param OrderEntity $order
     */
    public function createDraft(OrderEntity $order): void
    {
        $webhookOrder = [
            'orderNumber' => $order->getOrderNumber(),
            'deliveries' => [
                [
                    'shippingOrderAddressId' => $order->getDeliveries()->first()->getShippingOrderAddressId(),
                    'shippingMethod' => [
                        'name' => $order->getDeliveries()->first()->getShippingMethod()->getName(),
                    ]
                ]
            ],
            'addresses' => [],
            'customFields' => [
                'dhlparcel_shipping_servicepoint_id' =>
                    $order->getCustomFields()[ServicePointService::CUSTOM_FIELD_SERVICEPOINT_ID],
            ],
            'orderCustomer' => [
                'email' => $order->getOrderCustomer()->getEmail(),
            ]
        ];

        foreach ($order->getAddresses() as $address) {
            $webhookOrder['addresses'][] = [
                'id' => $address->getId(),
                'street' => $address->getStreet(),
                'additionalAddressLine1' => $address->getAdditionalAddressLine1(),
                'additionalAddressLine2' => $address->getAdditionalAddressLine2(),
                'company' => $address->getCompany(),
                'firstName' => $address->getFirstName(),
                'lastName' => $address->getLastName(),
                'phoneNumber' => $address->getPhoneNumber(),
                'country' => [
                    'iso' => $address->getCountry()->getIso(),
                ],
                'zipcode' => $address->getZipCode(),
                'city' => $address->getCity(),
            ];
        }

        $webhookUrl = $this->configService->getWebhookUrl();
        if (empty($webhookUrl)) {
            return;
        }

        try {
            $this->httpClient->post($webhookUrl, array(
                'headers' => [
                    'Content-Type'        => 'application/json',
                    'dhlparcel-connector' => 'shopware6'
                ],
                'body'    => json_encode($webhookOrder)
            ));
        } catch (\Exception $exception) {
            $this->logger->critical(
                'Failed to call webhook URL for order ' . $order->getOrderNumber(),
                ['exception' => $exception]
            );
        }
    }
}
