<?php declare(strict_types=1);

namespace DHLParcel\Shipping\Subscriber;

use DHLParcel\Shipping\Service\ConfigService;
use DHLParcel\Shipping\Service\ServicePointService;
use DHLParcel\Shipping\Service\WebhookService;
use Psr\Log\LoggerInterface;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Contracts\EventDispatcher\Event;
use Shopware\Core\Checkout\Cart\Event\CheckoutOrderPlacedEvent;

class NewOrderSubscriber implements EventSubscriberInterface
{
    /**
     * @var WebhookService
     */
    protected $webhookService;

    /**
     * @var ConfigService
     */
    protected $configService;

    /**
     * @var EntityRepository
     */
    private $orderRepository;

    /**
     * @var RequestStack
     */
    protected $requestStack;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * NewOrderSubscriber constructor.
     * @param ConfigService $configService
     * @param WebhookService $webhookService
     * @param EntityRepository $orderRepository
     * @param RequestStack $requestStack
     * @param LoggerInterface $logger
     */
    public function __construct(
        ConfigService $configService,
        WebhookService $webhookService,
        EntityRepository $orderRepository,
        RequestStack $requestStack,
        LoggerInterface $logger
    ) {
        $this->configService = $configService;
        $this->webhookService = $webhookService;
        $this->orderRepository = $orderRepository;
        $this->requestStack = $requestStack;
        $this->logger = $logger;
    }

    /**
     * @return array
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'state_enter.order_delivery.state.shipped' => 'onStateChange',
            'state_enter.order.state.completed'        => 'onStateChange',
            CheckoutOrderPlacedEvent::class            => 'onStateChange'
        ];
    }

    /**
     * @param Event  $event
     * @param string $eventName
     */
    public function onStateChange(Event $event, string $eventName): void
    {
        if (! $this->configService->isEnabled()) {
            return;
        }

        $order = $this->getOrder($event);

        // Save ServicePoint ID
        $shippingMethodIds = [];
        if (is_object($order->getDeliveries()) && method_exists($order->getDeliveries(), 'getShippingMethodIds')) {
            $shippingMethodIds = $order->getDeliveries()->getShippingMethodIds();
        }

        if ($eventName === CheckoutOrderPlacedEvent::class
            && in_array($this->configService->getServicePointShippingMethod(), $shippingMethodIds)
        ) {
            if ($this->saveServicePointId($event) === false) {
                return;
            }

            $order = $this->getOrder($event);
        }

        if (! $this->configService->isConfiguredForEvent($eventName)) {
            return;
        }

        try {
            $this->webhookService->createDraft($order);
        } catch (\Exception $exception) {
            $this->logger->critical(
                'Failed to create draft for order ' . $order->getOrderNumber(),
                ['exception' => $exception]
            );
        }
    }

    /**
     * @param Event $event
     * @return bool|null
     */
    protected function saveServicePointId(Event $event): bool
    {
        $customFieldsData = $this->requestStack->getCurrentRequest()->get('customFields');
        if (empty($customFieldsData) || !isset($customFieldsData[ServicePointService::CUSTOM_FIELD_SERVICEPOINT_ID])) {
            return false;
        }

        /** @var OrderEntity $order */
        $order = $event->getOrder();
        $orderCustomFields = [
            [
                'id' => $order->getId(),
                'customFields' => [
                    ServicePointService::CUSTOM_FIELD_SERVICEPOINT_ID =>
                        $customFieldsData[ServicePointService::CUSTOM_FIELD_SERVICEPOINT_ID]
                ]
            ]
        ];

        $orderUpdate = $this->orderRepository->update($orderCustomFields, $event->getContext());

        return empty($orderUpdate->getErrors());
    }

    protected function getOrder($event): OrderEntity
    {
        $criteria = (new Criteria([$event->getOrder()->getId()]))
            ->addAssociation('addresses')
            ->addAssociation('addresses.country')
            ->addAssociation('orderCustomer')
            ->addAssociation('lineItems')
            ->addAssociation('transactions')
            ->addAssociation('customFields')
            ->addAssociation('deliveries.shippingMethod')
            ->addAssociation('deliveries.positions.orderLineItem')
            ->addAssociation('deliveries.shippingOrderAddress.country')
            ->addAssociation('deliveries.shippingOrderAddress.countryState');

        return $this->orderRepository->search($criteria, $event->getContext())->get($event->getOrder()->getId());
    }
}
